<?php

namespace Daylight\Connector\Exact\Jobs;

use Daylight\Connector\Exact\Entities\Product;
use Daylight\Connector\Exact\Models\Unit;
use Daylight\Core\Models\TaxClass;
use Daylight\Core\Models\Variant;
use Illuminate\Bus\Batchable;
use Illuminate\Contracts\Queue\ShouldBeEncrypted;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class UpdateArticle implements ShouldBeEncrypted, ShouldBeUnique, ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function __construct(
        public Product $article
    ) {
        //
    }

    public function handle(): void
    {
        $variant = Variant::updateOrCreate([
            'external_id' => $this->article->id,
        ], [
            'tax_class_id' => $this->getTaxClass()?->id ?? 1,
            'unit_id' => $this->getUnit()?->id ?? null,
            'sku' => $this->article->code,
            'name' => $this->article->name,
            'weight' => $this->article->weight ? $this->article->weight / 1000 : 0,
            'stock' => $this->article->stock,
            'purchasable' => $this->article->canBePurchased(),
            'backorders' => $this->article->availableOnBackorder() ? 1 : 0,
            'external_product_group_id' => $this->article->itemGroupId,
        ]);

        $shippingClass = daylightModel('shipping_class')::query()
            ->whereDefault(1)
            ->first();

        if ($shippingClass && $variant) {
            $variant->shippingClasses()->sync([$shippingClass->id]);
        }
    }

    public function getUnit(): ?Unit
    {
        return Unit::query()
            ->whereCode($this->article->unit)
            ->first();
    }

    public function getTaxClass(): ?TaxClass
    {
        return TaxClass::query()
            ->whereExternalId($this->article->taxCode)
            ->first();
    }
}
