<?php

declare(strict_types=1);

namespace Daylight\Connector\Exact\Jobs;

use App\Models\User;
use Daylight\Connector\Exact\Entities\Order;
use Daylight\Connector\Exact\Entities\OrderLine;
use Daylight\Core\Models\Customer;
use Daylight\Core\Models\Enums\OrderStatus;
use Daylight\Core\Services\Countries;
use Illuminate\Bus\Batchable;
use Illuminate\Contracts\Queue\ShouldBeEncrypted;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;

class UpdateOrder implements ShouldBeEncrypted, ShouldBeUnique, ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /** @var Collection<int, mixed> */
    public Collection $taxClasses;

    public function __construct(public Order $order)
    {
        //
    }

    public function handle(): void
    {
        $customer = $this->getCustomer();

        if (! $customer) {
            return;
        }

        // Load tax classes if you want to derive the rate string from a tax class.
        $this->taxClasses = daylightModel('tax_class')::query()
            ->with('taxRates')
            ->get();

        // Split shipping lines (itemCode '03001') from normal lines.
        $allLines = $this->order->orderLines;
        $shippingLines = $allLines->filter(fn (OrderLine $l) => trim((string) $l->sku) === '03001');
        $normalLines = $allLines->reject(fn (OrderLine $l) => trim((string) $l->sku) === '03001');

        // Compute shipping (ex VAT) from lines: line-level, rounding AFTER qty.
        $shippingCents = $this->cents($shippingLines->sum(fn (OrderLine $l) => (float) $l->price * (int) $l->quantity));

        // Trust Exact for totals:
        $exactExclCents = $this->cents((float) $this->order->amountExcludingTax);
        $exactInclCents = $this->cents((float) $this->order->amountIncludingTax);

        // Subtotal excl. shipping, so subtotal + shipping == exactExclCents
        $subtotalCents = max(0, $exactExclCents - $shippingCents);

        // Base VAT from Exact (includes VAT over shipping when applicable)
        $taxCents = max(0, $exactInclCents - $exactExclCents);

        // Derive VAT rate string; if reverse charge code '7' is present, force 0% VAT and name "BTW verlegd".
        $rateString = $this->resolveRateString();

        if ($rateString === '0.0000') {
            $taxCents = 0;
        }

        // Minimal tax_breakdown per your schema.
        $taxBreakdown = collect([
            [
                'name' => 'BTW',
                'rate' => $rateString,
                'amount' => $taxCents,
            ],
        ]);

        $charges = $subtotalCents - $normalLines->sum(function (OrderLine $line) {
            return $this->cents((float) $line->price) * (int) $line->quantity;
        });

        $subtotalCents = $subtotalCents - $charges;

        DB::transaction(function () use ($customer, $charges, $subtotalCents, $shippingCents, $taxCents, $taxBreakdown) {
            $order = $customer->orders()->updateOrCreate(
                [
                    'external_id' => $this->order->id,
                    'number' => $this->order->orderNumber,
                ],
                [
                    'hash' => $this->getHash(),
                    'user_id' => $this->getUser()?->id,
                    'number' => $this->order->orderNumber,
                    'payment_method' => 'invoice',
                    'status' => $this->getStatus(),

                    // Subtotal EXCLUDING shipping; shipping stored separately.
                    'charges' => $charges,
                    'subtotal' => $subtotalCents,
                    'shipping' => $shippingCents,
                    'tax' => $taxCents,
                    'total' => $subtotalCents + $charges + $shippingCents + $taxCents,

                    'reference' => $this->order->reference,
                    'notes' => $this->order->notes,
                    'tax_breakdown' => $taxBreakdown,
                    'confirmation_send_at' => now(),
                    'draft' => 0,

                    'created_at' => $this->order->orderDate,
                ]
            );

            // Persist only NON-shipping lines as order items.
            $this->order->orderLines->each(function (OrderLine $orderLine) use ($order): void {
                if (trim((string) $orderLine->sku) === '03001') {
                    // Do NOT store shipping as an order item.
                    return;
                }

                $order->items()->updateOrCreate(
                    ['external_id' => $orderLine->id],
                    [
                        'variant_id' => daylightModel('variant')::query()
                            ->where('sku', $orderLine->sku)
                            ->value('id'),
                        'sku' => $orderLine->sku,
                        'name' => $orderLine->name,
                        'quantity' => (int) $orderLine->quantity,
                        // Unit price (ex VAT) in cents; line price is rounded later as needed in reporting.
                        'price' => $this->cents((float) $orderLine->price),
                        'order' => $orderLine->order,
                    ]
                );

                // Order addresses
            });

            // Store addresses
            $billingAddress = $customer->addresses->first();

            if (! $billingAddress) {
                $billingAddress = (object) $customer->only([
                    'name',
                    'address_line_1',
                    'address_line_2',
                    'postal_code',
                    'city',
                    'country_id',
                ]);
            }

            $order->addresses()->updateOrCreate(
                [
                    'type' => 'billing',
                ],
                [
                    'name' => $customer->name,
                    'address_line_1' => $billingAddress->address_line_1 ?? null,
                    'address_line_2' => $billingAddress->address_line_2 ?? null,
                    'postal_code' => $billingAddress->postal_code ?? null,
                    'city' => $billingAddress->city ?? null,
                    'country_id' => $billingAddress->country_id ?? null,
                    'vat_number' => $customer->vat_number,
                    'vat_number_verified_at' => $customer->vat_number && $billingAddress->country_id !== Countries::defaultCountry()->id ? now() : null,
                ]
            );

            $deliveryAddress = $customer->addresses()->where('external_id', $this->order->deliveryAddressId)->first();

            if (! $deliveryAddress) {
                $deliveryAddress = $billingAddress;
            }

            $order->addresses()->updateOrCreate([
                'type' => 'shipping',
            ], [
                'name' => $deliveryAddress->name,
                'address_line_1' => $deliveryAddress->address_line_1,
                'address_line_2' => $deliveryAddress->address_line_2,
                'postal_code' => $deliveryAddress->postal_code,
                'city' => $deliveryAddress->city,
                'country_id' => $deliveryAddress->country_id,
                'vat_number' => $customer->vat_number,
                'vat_number_verified_at' => $customer->vat_number && $deliveryAddress->country_id !== Countries::defaultCountry()->id ? now() : null,
            ]);

            return $order;
        });
    }

    /**
     * Determine VAT rate string:
     * - If any line has taxCode '7' (trimmed) => reverse charge (0.0000).
     * - Else try to resolve from tax_class; fall back to 21.0000.
     */
    private function resolveRateString(): string
    {
        // Reverse charge detection across lines
        $hasReverseCharge = $this->order->orderLines->contains(function (OrderLine $l): bool {
            return trim((string) $l->taxCode) === '7';
        });
        if ($hasReverseCharge) {
            return '0.0000';
        }

        // Otherwise derive from the first line's tax class (optional).
        $firstLine = $this->order->orderLines->first();
        if (! $firstLine) {
            return '21.0000';
        }

        $taxCode = trim((string) $firstLine->taxCode);
        $taxClass = $this->taxClasses->firstWhere('code', $taxCode);
        if (! $taxClass) {
            return '21.0000';
        }

        $taxRate = $taxClass->taxRates->firstWhere('country_id', Countries::defaultCountry()->id);
        if (! $taxRate) {
            return '21.0000';
        }

        return number_format((float) $taxRate->rate, 4, '.', '');
    }

    public function isDraft(): bool
    {
        // Optional consistency check against Exact inclusive amount:
        $orderLineTotal = $this->order->orderLines->sum(
            static fn (OrderLine $l): float => ($l->price * $l->quantity) + $l->taxAmount
        );

        return (int) round($orderLineTotal * 100) !== $this->cents((float) $this->order->amountIncludingTax);
    }

    public function getStatus(): OrderStatus
    {
        return match ($this->order->status) {
            12, 20 => OrderStatus::PROCESSING,
            21 => OrderStatus::COMPLETED,
            45 => OrderStatus::CANCELLED,
            default => OrderStatus::PENDING,
        };
    }

    public function getHash(): string
    {
        return md5((string) json_encode($this->order, JSON_THROW_ON_ERROR));
    }

    public function getCustomer(): ?Customer
    {
        return daylightModel('customer')::query()
            ->whereExternalId($this->order->customerId)
            ->with([
                'addresses' => function ($query) {
                    $query->whereType('billing')->orderBy('default', 'desc')->limit(1);
                },
            ])
            ->first();
    }

    public function getUser(): ?User
    {
        return User::query()
            ->whereExternalId($this->order->authorId)
            ->first();
    }

    // ---- helpers ----
    private function cents(float $amount): int
    {
        return (int) round($amount * 100, 0);
    }

    private function euros(int $cents): float
    {
        return $cents / 100;
    }
}
