# Extending Daylight
- [Introduction](#introduction)
- [Usage of icons](#usage-of-icons)
- [Extending the admin dashboard](#dashboard)
  - [Creating a new admin page](#creating-a-new-admin-page)
  - [Creating a new dashboard widget](#creating-a-new-dashboard-widget)

<a name="introduction"></a>
## Introduction

Daylight is designed to be easily extensible, allowing you to add new features and functionality through service providers, custom routes, and more. Below are some examples of how to extend Daylight.

### Translations (key-based, EN/NL) and JS i18n

- Publish vendor translations to override:

```bash
php artisan vendor:publish --tag=daylight-translations
```

- PHP/Blade usage (Laravel-style, namespaced):

```php
<?= __('daylight::general.save') ?>
```

- Vue usage (after the admin app initializes i18n):

```vue
{{ __('daylight::general.save') }}
```

- Keys are of the form `daylight::group.key`. Group files live under `resources/lang/{locale}/{group}.php` in the package, and can be overridden in `resources/lang/vendor/daylight/{locale}/{group}.php` after publishing.

- The admin app loads translations asynchronously via `GET {admin}/api/translations/{locale}` and caches them in the browser. The following globals are injected in the admin HTML:
  - `window.daylight.locale`
  - `window.daylight.fallbackLocale`
  - `window.daylight.translationsUrl`

You can add more groups exposed to JS via `config/daylight.php` in `translations.groups`.

#### Usage in your application

All strings in the Daylight package are already translated. If you need to add custom translations:

- Add keys to the appropriate group files: `resources/lang/en/{group}.php` and `resources/lang/nl/{group}.php`
- Use the same `daylight::group.key` format for consistency
- Available groups: `general`, `store`, `content`, `settings`, `media`

## Usage of icons

Daylight uses Blade Icons with the **Heroicons** set. You can use these icons in your views by referencing them directly. More information can be found in the [Blade Icons documentation](https://blade-ui-kit.com/blade-icons).

For a list of available icons, you can refer to the [Blade Icons Overview](https://blade-ui-kit.com/blade-icons#search). Make sure to select 'Heroicons' from the icon sets.


<a name="extending-the-admin-dashboard"></a>
## Extending the admin dashboard

<a name="creating-a-new-admin-page"></a>
### Creating a new admin page

Within the register method of your service provider:

```php
use Daylight\Core\Modules\Admin\Registries\NavigationRegistry;
use Daylight\Core\Modules\Admin\Entities\Page;

NavigationRegistry::addPage(
    Page::create()
        ->setOrder(200)
        ->setKey('store')
        ->setLabel('Store')
        ->setPermission('read:orders')
        ->setRoute('daylight.store.orders.index')
        ->setIcon('o-shopping-bag')
        ->setChildren([
            Page::create()
                ->setKey('orders')
                ->setLabel('Orders')
                ->setPermission('read:orders')
                ->setRoute('daylight.store.orders.index')
                ->setOrder(100),
        ])
)
```

<a href="#creating-a-new-dashboard-widget"></a>
### Creating a new dashboard widget

Within the register method of your service provider:

```php
use Daylight\Core\Modules\Admin\Registries\WidgetRegistry;
use Daylight\Core\Modules\Admin\Entities\Widget;

WidgetRegistry::addWidget(
    new ExampleWidget()
)
```
Rendering the widget:
```php
use Illuminate\View\View;
use Daylight\Core\Models\Customer;

class ExampleWidget extends Widget
{
    public function render(): View
    {
        return view('daylight::widgets.customer-count', [
            'customers' => Customer::count(),
        ]);
    }
}
```

