@extends('daylight::layouts.app')

@section('title', __('Edit shipping zone'))

@section('content')
    <div x-data="{ showConfirmDeleteModal: false }">
        <x-daylight::page-header :title="__('Edit :page', ['page' => $shippingZone->name])">
            @can('update:advanced-settings')
                <x-daylight::buttons.danger @click.prevent="showConfirmDeleteModal = true">
                    {{ __('Delete shipping zone') }}
                </x-daylight::buttons.danger>
            @endif
        </x-daylight::page-header>

        <x-daylight::session-messages/>

        <form method="post" action="{{ route('daylight.settings.shipping-zones.update', $shippingZone) }}">
            @csrf
            @method('PUT')

            <x-daylight::input.group
                :label="__('Name')"
                :error="$errors->first('name')"
                for="code"
            >
                <x-daylight::input.text
                    name="name"
                    id="name"
                    value="{{ old('name', $shippingZone->name) }}"
                    :disabled="!auth()->user()->can('update:advanced-settings')"
                />
            </x-daylight::input.group>

            <x-daylight::input.group
                :label="__('Default')"
                :error="$errors->first('default')"
                for="default"
            >
                <div x-data="{ selected: {{ old('default', $shippingZone->default) ? '1' : '0' }}}">
                    <div class="flex items-center mt-1">
                        <button :class="{ 'bg-sky-600': selected, 'bg-gray-200': !selected }"
                                @click.prevent="selected = (selected === 0 ? 1 : 0)" type="button"
                                class="relative inline-flex h-6 w-11 flex-shrink-0 cursor-pointer rounded-full border-2 border-transparent bg-gray-200 transition-colors duration-200 ease-in-out focus:outline-none focus:ring-2 focus:ring-indigo-600 focus:ring-offset-2"
                                role="switch" aria-checked="false" aria-labelledby="annual-billing-label">
                            <span :class="{ 'translate-x-5': selected, 'translate-x-0': !selected }" aria-hidden="true"
                                  class="pointer-events-none inline-block h-5 w-5 translate-x-0 transform rounded-full bg-white shadow ring-0 transition duration-200 ease-in-out"></span>
                        </button>
                        <span class="ml-3 text-sm font-medium text-gray-900"
                              id="annual-billing-label">{{ __('Default') }}</span>
                    </div>

                    <input type="hidden" name="default" x-model="selected">
                </div>
            </x-daylight::input.group>

            <x-daylight::input.group
                :label="__('Max amount')"
                :error="$errors->first('max_amount')"
                :help-text="__('The maximum amount of money that can be charged for this shipping zone.')"
                for="max_amount"
            >
                <x-daylight::input.text
                    type="number"
                    name="max_amount"
                    id="max_amount"
                    value="{{ old('max_amount', $shippingZone->max_amount / 100) }}"
                />
            </x-daylight::input.group>

            <x-daylight::input.group
                :label="__('Countries')"
                :error="$errors->first('countries')"
                for="countries"
            >
                <div class="grid grid-cols-3 gap-x-4 gap-y-1">
                    @foreach($countries as $country)
                        <div class="relative flex items-start">
                            <div class="flex h-6 items-center">
                                <input
                                    {{ !auth()->user()->can('update:settings') ? 'disabled' : '' }}
                                    id="{{ $country->id }}"
                                    name="countries[]"
                                    type="checkbox"
                                    class="h-4 w-4 rounded border-gray-300 text-sky-600 !ring-0 !outline-0"
                                    value="{{ $country->id }}"
                                    {{ in_array($country->id, old('countries', $shippingZone->countries->pluck('id')->toArray())) ? 'checked' : '' }}
                                >
                            </div>
                            <div class="ml-3 text-sm leading-6">
                                <label for="{{ $country->id }}" class="text-gray-900"> {{ $country->name }}</label>
                            </div>
                        </div>
                    @endforeach
                </div>
            </x-daylight::input.group>

            <x-daylight::input.group
                :label="__('Rules')"
                :error="$errors->first('rules')"
                for="code"
            >
                <div x-data="{ shippingClass: {{ $shippingClasses->first()['class']->id }} }">
                    <div>
                        <div class="sm:hidden">
                            <label for="tabs" class="sr-only">Select a tab</label>
                            <!-- Use an "onChange" listener to redirect the user to the selected tab URL. -->
                            <select x-model="shippingClass" id="tabs"
                                    class="block w-full rounded-md border-gray-300 focus:border-indigo-500 focus:ring-indigo-500">
                                @foreach($shippingClasses as $shippingClass)
                                    <option
                                        value="{{ $shippingClass['class']->id }}">{{ $shippingClass['class']->name }}</option>
                                @endforeach
                            </select>
                        </div>
                        <div class="hidden sm:block">
                            <nav class="flex space-x-4" aria-label="Tabs">
                                @foreach($shippingClasses as $shippingClass)
                                    <a href="#"
                                       @click.prevent="shippingClass = {{ $shippingClass['class']->id }}"
                                       class="rounded-md px-3 py-2 text-sm font-medium text-gray-500 hover:text-gray-700"
                                       :class="{
                                                   'bg-sky-100 text-sky-700': shippingClass == {{ $shippingClass['class']->id }},
                                                   'text-gray-500 hover:text-gray-700': shippingClass != {{ $shippingClass['class']->id }}
                                               }">
                                        {{ $shippingClass['class']->name }}
                                    </a>
                                @endforeach
                            </nav>
                        </div>
                    </div>

                    @foreach($shippingClasses as $shippingClass)
                        <div x-show="shippingClass == {{ $shippingClass['class']->id }}" x-cloak>
                            @if($shippingClass['class']->cumulative)
                                <div class="rounded-md bg-yellow-50 mt-4 p-4">
                                    <div class="flex">
                                        <div class="flex-shrink-0">
                                            <svg class="h-5 w-5 text-yellow-400" viewBox="0 0 20 20" fill="currentColor"
                                                 aria-hidden="true">
                                                <path fill-rule="evenodd"
                                                      d="M8.485 2.495c.673-1.167 2.357-1.167 3.03 0l6.28 10.875c.673 1.167-.17 2.625-1.516 2.625H3.72c-1.347 0-2.189-1.458-1.515-2.625L8.485 2.495zM10 5a.75.75 0 01.75.75v3.5a.75.75 0 01-1.5 0v-3.5A.75.75 0 0110 5zm0 9a1 1 0 100-2 1 1 0 000 2z"
                                                      clip-rule="evenodd"/>
                                            </svg>
                                        </div>
                                        <div class="ml-3">
                                            <h3 class="text-sm font-medium text-yellow-800">{{ __('Cumulative rule') }}</h3>
                                            <div class="mt-2 text-sm text-yellow-700">
                                                <p>{{ __('This is cumulative rule, which means that the costs of specified rules with by multiplied by the quantity of the product. When a customer purchases 3 items with this class attached, the costs below will be charged 3 times.') }}</p>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            @endif

                            <div class="grid grid-cols-3 gap-4 mt-6 text-gray-500 text-sm">
                                <div>
                                    {{ __('Minimum weight (Grams)') }}
                                </div>

                                <div>
                                    {{ __('Maximum weight (Grams)') }}
                                </div>

                                <div>
                                    {{ __('Price') }}
                                </div>
                            </div>

                            <div class="divide-y space-y-2 mt-2" x-data='{
                                    rules:
                                        {{
                                            old('rules.' . $shippingClass['class']->id)
                                                ? json_encode(old('rules.' . $shippingClass['class']->id))
                                                : json_encode($shippingClass['rules']->map(function($rule) {
                                                    return [
                                                        ...$rule->only(['id', 'min_weight', 'max_weight', 'price']),
                                                        ...['price' => $rule->price / 100]
                                                    ];
                                                })->values()->toArray())
                                        }}
                               }'
                            >
                                <template x-for="(rule, index) in rules" :key="`{{ $shippingClass['class']->id  }}-${index}`">
                                    <div class="grid grid-cols-3 py-1 gap-4">
                                        <input type="hidden" x-bind:name="`rules[{{ $shippingClass['class']->id  }}][${index}][id]`" x-bind:value="rule.id">

                                        <x-daylight::input.text
                                            type="number"
                                            x-bind:name="`rules[{{ $shippingClass['class']->id  }}][${index}][min_weight]`"
                                            x-model="rule.min_weight"
                                            placeholder="{{ __('Optional') }}"
                                            class="w-full mt-2"/>

                                        <x-daylight::input.text
                                            type="number"
                                            x-bind:name="`rules[{{ $shippingClass['class']->id  }}][${index}][max_weight]`"
                                            x-model="rule.max_weight"
                                            placeholder="{{ __('Optional') }}"
                                            class="w-full mt-2"/>

                                        <div class="mt-2 flex items-center justify-between gap-4">
                                            <x-daylight::input.text
                                                required
                                                type="number"
                                                x-bind:name="`rules[{{ $shippingClass['class']->id  }}][${index}][price]`"
                                                x-model="rule.price"
                                                class="w-full"/>

                                            <button
                                                class="min-w-8 flex-grow w-8 h-8 bg-red-600 text-white flex items-center justify-center rounded-full hover:bg-red-800"
                                                @click.prevent="rules.splice(index, 1)"
                                            >
                                                <svg class="w-4 h-4" data-slot="icon" fill="none" stroke-width="1.5" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg" aria-hidden="true">
                                                    <path stroke-linecap="round" stroke-linejoin="round" d="m14.74 9-.346 9m-4.788 0L9.26 9m9.968-3.21c.342.052.682.107 1.022.166m-1.022-.165L18.16 19.673a2.25 2.25 0 0 1-2.244 2.077H8.084a2.25 2.25 0 0 1-2.244-2.077L4.772 5.79m14.456 0a48.108 48.108 0 0 0-3.478-.397m-12 .562c.34-.059.68-.114 1.022-.165m0 0a48.11 48.11 0 0 1 3.478-.397m7.5 0v-.916c0-1.18-.91-2.164-2.09-2.201a51.964 51.964 0 0 0-3.32 0c-1.18.037-2.09 1.022-2.09 2.201v.916m7.5 0a48.667 48.667 0 0 0-7.5 0"></path>
                                                </svg>
                                            </button>
                                        </div>
                                    </div>
                                </template>

                                <div x-show="rules.length === 0">
                                    <div class="rounded-md bg-blue-50 p-4 my-4">
                                        <div class="flex">
                                            <div class="flex-shrink-0">
                                                <svg class="h-5 w-5 text-blue-400" viewBox="0 0 20 20" fill="currentColor" aria-hidden="true">
                                                    <path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a.75.75 0 000 1.5h.253a.25.25 0 01.244.304l-.459 2.066A1.75 1.75 0 0010.747 15H11a.75.75 0 000-1.5h-.253a.25.25 0 01-.244-.304l.459-2.066A1.75 1.75 0 009.253 9H9z" clip-rule="evenodd" />
                                                </svg>
                                            </div>
                                            <div class="ml-3 flex-1 md:flex md:justify-between">
                                                <p class="text-sm text-blue-700">{{ __('No rules configured for this shipping class.') }}</p>
                                            </div>
                                        </div>
                                    </div>

                                </div>

                                <x-daylight::buttons.primary
                                    class="mt-4"
                                    @click.prevent="rules.push({id: null, min_price: null, max_price: null, price: null})">
                                    {{ __('Add rule') }}
                                </x-daylight::buttons.primary>
                            </div>
                        </div>
                    @endforeach
                </div>
            </x-daylight::input.group>

            @can('update:advanced-settings')
                <div class="mt-6 flex items-center justify-end gap-x-3">
                    <x-daylight::buttons.secondary href="{{ route('daylight.settings.shipping-zones.index') }}" element="a">
                        {{ __('Cancel') }}
                    </x-daylight::buttons.secondary>

                    <x-daylight::buttons.primary type="submit">
                        {{ __('Save') }}
                    </x-daylight::buttons.primary>
                </div>
            @endcan
        </form>

        <x-daylight::modal.confirm-delete-modal>
            <x-slot name="content">
                <h3 class="text-base font-semibold leading-6 text-gray-900"
                    id="modal-title">{{ __('Delete :object', ['object' => __('shipping zone')]) }}</h3>
                <div class="mt-2">
                    <p class="text-sm text-gray-500">
                        {{ __('Are you sure you want to delete this :object? This action cannot be undone.', ['object' => __('shipping zone')]) }}
                    </p>
                </div>
            </x-slot>

            <x-slot name="footer">
                <form method="post" action="{{ route('daylight.settings.shipping-zones.destroy', $shippingZone) }}">
                    @csrf
                    @method('DELETE')

                    <button type="button"
                            class="mt-3 inline-flex w-full justify-center rounded-md bg-white px-3 py-2 text-sm font-semibold text-gray-900 shadow-sm ring-1 ring-inset ring-gray-300 hover:bg-gray-100 sm:mt-0 sm:w-auto"
                            @click.prevent="showConfirmDeleteModal = false">
                        {{ __('Cancel') }}
                    </button>

                    <button type="submit"
                            class="inline-flex w-full justify-center rounded-md bg-red-600 px-3 py-2 text-sm font-semibold text-white shadow-sm hover:bg-red-500 sm:ml-2 sm:w-auto">
                        {{ __('Delete') }}
                    </button>
                </form>
            </x-slot>
        </x-daylight::modal.confirm-delete-modal>
    </div>
@endsection
