<?php

use Daylight\Core\DaylightServiceProvider;

if (! function_exists('currentDaylightVersion')) {
    function currentDaylightVersion(): string
    {
        return DaylightServiceProvider::CURRENT_VERSION ?? 'Unknown';
    }
}

if (! function_exists('translatedRoute')) {
    function translatedRoute($name, $parameters = [], $lang = null, $absolute = true)
    {
        if ($lang && in_array($lang, config('multilingualism.available_locales'))) {
            return app('url')->route($lang.'_'.$name, $parameters, $absolute);
        }

        $locale_prefix = config('multilingualism.locale_prefix');

        if (empty($locale_prefix)) {
            $locale_prefix = app()->getLocale();
        }

        if (empty($locale_prefix)) {
            $locale_prefix = config('app.fallback_locale');
        }

        return app('url')->route($locale_prefix.'_'.$name, $parameters, $absolute);
    }
}

if (! function_exists('daylightAsset')) {
    function daylightAsset(string $path): string
    {
        static $manifests = [];

        $reloadPath = public_path('vendor/daylight/hot');

        if (file_exists($reloadPath)) {
            return file_get_contents($reloadPath).'/'.$path;
        }

        $manifestPath = public_path('vendor/daylight/build/manifest.json');

        if (! isset($manifests[$manifestPath])) {
            if (! file_exists($manifestPath)) {
                if (app()->runningInConsole()) {
                    return false;
                }

                throw new RuntimeException("Vite manifest not found at [$manifestPath].");
            }

            $manifests[$manifestPath] = json_decode(file_get_contents($manifestPath), true);
        }

        $manifest = $manifests[$manifestPath];

        if (! isset($manifest[$path])) {
            return asset("vendor/daylight/build/{$manifest[$path]['file']}");
        }

        return asset("vendor/daylight/build/{$manifest[$path]['file']}");
    }
}

if (! function_exists('formatPrice')) {
    function formatPrice(mixed $price, int $decimals = 2): string
    {
        if (app()->getLocale() == 'fr') {
            return number_format($price, $decimals, ',', ' ').' €';
        }

        if (in_array(app()->getLocale(), ['de', 'es'])) {
            return number_format($price, $decimals, ',', '.').' €';
        }

        return '€ '.number_format($price, $decimals, ',', '.');
    }
}

if (! function_exists('formatBytes')) {
    function formatBytes($size, $precision = 2): string
    {
        if ($size > 0) {
            $size = (int) $size;
            $base = log($size) / log(1024);
            $suffixes = [' bytes', ' KB', ' MB', ' GB', ' TB'];

            return round(pow(1024, $base - floor($base)), $precision).$suffixes[floor($base)];
        } else {
            return $size;
        }
    }
}

if (! function_exists('formatNumber')) {
    function formatNumber(mixed $amount, int $decimals = 0)
    {
        if (app()->getLocale() == 'fr') {
            return number_format($amount, $decimals, ',', ' ');
        }

        return number_format($amount, $decimals, ',', '.');
    }
}

if (! function_exists('getFileExtension')) {
    function getFileExtension(string $filename): string
    {
        return pathinfo($filename, PATHINFO_EXTENSION);
    }
}
