<?php

namespace Daylight\Core\Models;

use Daylight\Core\Models\Concerns\IsRelatable;
use Daylight\Core\Models\Contracts\Relatable;
use Daylight\Core\Modules\Multilingualism\Contracts\HasTranslations;
use Daylight\Core\Modules\Multilingualism\Models\Concerns\InteractsWithTranslations;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Support\Collection;
use Staudenmeir\LaravelAdjacencyList\Eloquent\HasRecursiveRelationships;

class Category extends Model implements HasTranslations, Relatable
{
    use HasRecursiveRelationships;
    use InteractsWithTranslations;
    use IsRelatable;

    protected $guarded = [];

    protected static function booted(): void
    {
        // Update nested slugs when category hierarchy changes
        static::saved(function (Category $category) {
            if ($category->wasChanged('parent_id')) {
                $category->load('translations');
                foreach ($category->translations as $translation) {
                    $translation->updateNestedSlug();
                    $translation->saveQuietly();
                    $translation->updateDescendantNestedSlugs();
                }
            }
        });
    }

    public function translatedFields(): array
    {
        return ['name', 'slug', 'meta_title', 'meta_description', 'robots'];
    }

    public function children(): HasMany
    {
        return $this->hasMany(daylightModel('category'), 'parent_id');
    }

    public function childrenRecursive(): HasMany
    {
        return $this->hasMany(daylightModel('category'), 'parent_id')->with('childrenRecursive');
    }

    public function attributes(): BelongsToMany
    {
        return $this->belongsToMany(daylightModel('attribute'));
    }

    public function products(): BelongsToMany
    {
        return $this->belongsToMany(daylightModel('product'));
    }

    public function translations(): HasMany
    {
        return $this->hasMany(daylightModel('category_translation'));
    }

    public function defaultTranslation(): HasOne
    {
        return $this->hasOne(daylightModel('category_translation'))
            ->where('locale', config('multilingualism.default_locale'));
    }

    // Override IsRelatable trait for translatable model
    public static function searchForRelation(string $query, array $excludeIds = [], int $limit = 10): Collection
    {
        $queryBuilder = static::query();

        if (! empty($excludeIds)) {
            $queryBuilder->whereNotIn('id', $excludeIds);
        }

        // Search in translations
        $queryBuilder->whereHas('translations', function ($q) use ($query) {
            $q->where('name', 'LIKE', "%{$query}%");
        });

        return $queryBuilder->limit($limit)
            ->get()
            ->map(fn ($item) => $item->formatForRelation());
    }

    public function formatForRelation(): array
    {
        return [
            'id' => $this->id,
            'name' => $this->name,
            'type' => 'Category',
        ];
    }
}
