<?php

namespace Daylight\Core\Modules\Admin;

use Daylight\Core\Modules\Admin\Entities\Divider;
use Daylight\Core\Modules\Admin\Entities\Page;
use Daylight\Core\Modules\Admin\Entities\Script;
use Daylight\Core\Modules\Admin\Entities\Shortcut;
use Daylight\Core\Modules\Admin\Entities\Stylesheet;
use Daylight\Core\Modules\Admin\Providers\AssetServiceProvider;
use Daylight\Core\Modules\Admin\Providers\NavigationServiceProvider;
use Daylight\Core\Modules\Admin\Providers\ShortcutServiceProvider;
use Daylight\Core\Modules\Admin\Providers\WidgetServiceProvider;
use Daylight\Core\Modules\Admin\Registries\AssetRegistry;
use Daylight\Core\Modules\Admin\Registries\NavigationRegistry;
use Daylight\Core\Modules\Admin\Registries\ShortcutRegistry;
use Daylight\Core\Modules\Admin\Registries\WidgetRegistry;
use Daylight\Core\Modules\Admin\Widgets\CustomerCount;
use Daylight\Core\Modules\Admin\Widgets\OrderCount;
use Daylight\Core\Modules\Admin\Widgets\ProductCount;
use Illuminate\Support\ServiceProvider;

class AdminServiceProvider extends ServiceProvider
{
    public function register(): void
    {
        $this->app->register(AssetServiceProvider::class);
        $this->app->register(NavigationServiceProvider::class);
        $this->app->register(WidgetServiceProvider::class);
        $this->app->register(ShortcutServiceProvider::class);

        $this
            ->registerAssets()
            ->registerWidgets();
    }

    public function boot(): void
    {
        $this
            ->registerAdminNavigation()
            ->registerShortcuts();
    }

    public function registerAssets(): self
    {
        AssetRegistry::addAsset(
            asset: Stylesheet::create()->setPath(daylightAsset('resources/assets/css/app.scss'))
        );

        AssetRegistry::addAsset(
            asset: Script::create()->setPath(daylightAsset('resources/assets/js/app.js'))->setDefer(true)
        );

        return $this;
    }

    public function registerAdminNavigation(): self
    {
        NavigationRegistry::addPage(
            Page::create()
                ->setOrder(100)
                ->setKey('dashboard')
                ->setLabel('daylight::general.dashboard')
                ->setPermission('view:administration')
                ->setIcon('o-home')
                ->setRoute('daylight.index')
        );

        NavigationRegistry::addPage(
            Page::create()
                ->setOrder(200)
                ->setKey('store')
                ->setLabel('daylight::general.store')
                ->setPermission('read:orders')
                ->setRoute('daylight.store.orders.index')
                ->setIcon('o-shopping-bag')
                ->setChildren([
                    Page::create()
                        ->setKey('orders')
                        ->setLabel('daylight::store.orders')
                        ->setPermission('read:orders')
                        ->setRoute('daylight.store.orders.index')
                        ->setOrder(100),
                    Page::create()
                        ->setKey('customers')
                        ->setLabel('daylight::store.customers')
                        ->setPermission('read:customers')
                        ->setRoute('daylight.store.customers.index')
                        ->setOrder(200),
                    Page::create()
                        ->setKey('customer_groups')
                        ->setLabel('daylight::store.customer_groups')
                        ->setPermission('read:customer-groups')
                        ->setRoute('daylight.store.customer-groups.index')
                        ->setOrder(300),
                    Page::create()
                        ->setKey('users')
                        ->setLabel('daylight::store.users')
                        ->setPermission('read:users')
                        ->setRoute('daylight.store.users.index')
                        ->setOrder(400),
                    Page::create()
                        ->setKey('products')
                        ->setLabel('daylight::store.products')
                        ->setPermission('read:products')
                        ->setRoute('daylight.store.products.index')
                        ->setOrder(500),
                    Page::create()
                        ->setKey('variants')
                        ->setLabel('daylight::store.variants')
                        ->setPermission('read:variants')
                        ->setRoute('daylight.store.variants.index')
                        ->setOrder(500),
                    Page::create()
                        ->setKey('brands')
                        ->setLabel('daylight::store.brands')
                        ->setPermission('read:brands')
                        ->setRoute('daylight.store.brands.index')
                        ->setOrder(600),
                    Page::create()
                        ->setKey('categories')
                        ->setLabel('daylight::store.categories')
                        ->setPermission('read:categories')
                        ->setRoute('daylight.store.categories.index')
                        ->setOrder(700),
                    Page::create()
                        ->setKey('labels')
                        ->setLabel('daylight::store.labels')
                        ->setPermission('read:labels')
                        ->setRoute('daylight.store.labels.index')
                        ->setOrder(800),
                    Page::create()
                        ->setKey('attributes')
                        ->setLabel('daylight::store.attributes')
                        ->setPermission('read:attributes')
                        ->setRoute('daylight.store.attributes.index')
                        ->setOrder(900),
                ])
        );

        NavigationRegistry::addPage(
            Page::create()
                ->setKey('content')
                ->setLabel('daylight::general.content')
                ->setIcon('o-document-duplicate')
                ->setPermission('read:pages')
                ->setRoute('daylight.content.pages.index')
                ->setOrder(300)
                ->setChildren([
                    Page::create()
                        ->setKey('pages')
                        ->setLabel('daylight::content.pages')
                        ->setPermission('read:pages')
                        ->setRoute('daylight.content.pages.index')
                        ->setOrder(100),
                ]),
        );

        NavigationRegistry::addPage(
            Page::create()
                ->setOrder(1000)
                ->setKey('settings')
                ->setLabel('daylight::settings.settings')
                ->setRoute('daylight.settings.general.index')
                ->setPermission('read:settings')
                ->setIcon('o-cog-6-tooth')
                ->setChildren([
                    Page::create()
                        ->setKey('general')
                        ->setLabel('daylight::settings.general')
                        ->setPermission('read:settings')
                        ->setRoute('daylight.settings.general.index')
                        ->setOrder(100),
                    Divider::create()
                        ->setLabel('daylight::general.store')
                        ->setOrder(150),
                    Page::create()
                        ->setKey('countries')
                        ->setLabel('daylight::settings.countries')
                        ->setPermission('read:settings')
                        ->setRoute('daylight.settings.countries.index')
                        ->setOrder(200),
                    Page::create()
                        ->setKey('shipping')
                        ->setLabel('daylight::general.shipping')
                        ->setPermission('read:settings')
                        ->setRoute('daylight.settings.shipping-zones.index')
                        ->setOrder(300),
                    Divider::create()
                        ->setLabel('daylight::settings.advanced_settings')
                        ->setOrder(350),
                    Page::create()
                        ->setKey('tax_classes')
                        ->setLabel('daylight::settings.tax_classes')
                        ->setPermission('read:advanced-settings')
                        ->setRoute('daylight.settings.tax-classes.index')
                        ->setOrder(400),
                    Page::create()
                        ->setKey('shipping_classes')
                        ->setLabel('daylight::settings.shipping_classes')
                        ->setPermission('read:advanced-settings')
                        ->setRoute('daylight.settings.shipping-classes.index')
                        ->setOrder(500),
                    Page::create()
                        ->setKey('roles_permissions')
                        ->setLabel('daylight::settings.roles_permissions')
                        ->setPermission('read:advanced-settings')
                        ->setRoute('daylight.settings.roles.index')
                        ->setOrder(600),
                ])
        );

        return $this;
    }

    public function registerWidgets(): self
    {
        WidgetRegistry::addWidget(
            new CustomerCount
        );

        WidgetRegistry::addWidget(
            new OrderCount
        );

        WidgetRegistry::addWidget(
            new ProductCount
        );

        return $this;
    }

    public function registerShortcuts(): self
    {
        ShortcutRegistry::addShortcut(
            Shortcut::create()
                ->setKey('orders')
                ->setColor('bg-pink-500')
                ->setIcon('o-shopping-cart')
                ->setLabel('daylight::general.shortcuts.view_latest_orders')
                ->setDescription('daylight::general.shortcuts.view_latest_orders_desc')
                ->setRoute('daylight.store.orders.index')
                ->setPermission('read:orders')
        );

        ShortcutRegistry::addShortcut(
            Shortcut::create()
                ->setKey('customer-groups')
                ->setColor('bg-yellow-500')
                ->setIcon('o-users')
                ->setLabel('daylight::general.shortcuts.manage_customer_groups')
                ->setDescription('daylight::general.shortcuts.manage_customer_groups_desc')
                ->setRoute('daylight.store.customer-groups.index')
                ->setPermission('read:customer-groups')
        );

        ShortcutRegistry::addShortcut(
            Shortcut::create()
                ->setKey('create-product')
                ->setColor('bg-green-500')
                ->setIcon('o-document')
                ->setLabel('daylight::general.shortcuts.create_new_product')
                ->setDescription('daylight::general.shortcuts.create_new_product_desc')
                ->setRoute('daylight.store.products.create')
                ->setPermission('create:products')
        );

        ShortcutRegistry::addShortcut(
            Shortcut::create()
                ->setKey('variants')
                ->setColor('bg-blue-500')
                ->setIcon('o-document-duplicate')
                ->setLabel('daylight::general.shortcuts.browse_variants')
                ->setDescription('daylight::general.shortcuts.browse_variants_desc')
                ->setRoute('daylight.store.variants.index')
                ->setPermission('read:variants')
        );

        ShortcutRegistry::addShortcut(
            Shortcut::create()
                ->setKey('attributes')
                ->setColor('bg-blue-500')
                ->setIcon('o-funnel')
                ->setLabel('daylight::general.shortcuts.manage_attributes')
                ->setDescription('daylight::general.shortcuts.manage_attributes_desc')
                ->setRoute('daylight.store.attributes.index')
                ->setPermission('read:attributes')
        );

        return $this;
    }
}
