<div 
    x-data="{
        selectedFiles: [],
        isDragging: false,
        
        init() {
            // Listen for when Livewire finishes processing
            this.$watch('$wire.uploadingFiles', (files) => {
                if (files.length > 0) {
                    this.selectedFiles = files;
                }
            });
        },
        
        handleFileSelect(event) {
            const files = event.target.files;
            this.showFilesImmediately(files);
        },
        
        handleDrop(event) {
            this.isDragging = false;
            const files = event.dataTransfer.files;
            
            // Set files to input element so Livewire can process them
            this.$refs.fileInput.files = files;
            
            this.showFilesImmediately(files);
        },
        
        showFilesImmediately(files) {
            if (!files || files.length === 0) return;
            
            // Immediately show files in UI
            this.selectedFiles = Array.from(files).map(file => ({
                name: file.name,
                size: file.size,
                status: 'pending',
                error: null
            }));
        }
    }"
    x-on:uploads-complete.window="$wire.closeModalAfterUpload()"
>
    <!-- Upload Modal -->
    <template x-teleport="body">
        <div x-cloak x-show="$wire.showModal" class="fixed inset-0 z-50" style="display: none;">
            <div class="fixed inset-0 overflow-y-auto px-4 py-6 sm:px-0 z-50">
                <div class="flex items-center min-h-full">
                    <!-- Backdrop -->
                    <div 
                        class="fixed inset-0 bg-gray-500 opacity-75 transition-opacity"
                        x-on:click="$wire.closeModal()"
                        x-show="$wire.showModal"
                        x-transition:enter="ease-out duration-300"
                        x-transition:enter-start="opacity-0"
                        x-transition:enter-end="opacity-100"
                        x-transition:leave="ease-in duration-200"
                        x-transition:leave-start="opacity-100"
                        x-transition:leave-end="opacity-0"
                    ></div>

                    <!-- Modal Content -->
                    <div
                        class="relative bg-white rounded-lg shadow-xl transform transition-all sm:w-full sm:mx-auto sm:max-w-2xl z-10"
                        x-show="$wire.showModal"
                        x-transition:enter="ease-out duration-300"
                        x-transition:enter-start="opacity-0 translate-y-4 sm:translate-y-0 sm:scale-95"
                        x-transition:enter-end="opacity-100 translate-y-0 sm:scale-100"
                        x-transition:leave="ease-in duration-200"
                        x-transition:leave-start="opacity-100 translate-y-0 sm:scale-100"
                        x-transition:leave-end="opacity-0 translate-y-4 sm:translate-y-0 sm:scale-95"
                    >
                        <div class="bg-sky-600 px-6 py-4 rounded-t-lg">
                            <div class="flex items-start justify-between">
                                <h2 class="text-lg font-semibold text-white">{{ __('daylight::media.upload_files') }}</h2>
                                <button x-on:click.prevent="$wire.closeModal(); selectedFiles = [];" type="button" class="text-white hover:text-opacity-75">
                                    <svg class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor">
                                        <path stroke-linecap="round" stroke-linejoin="round" d="M6 18L18 6M6 6l12 12" />
                                    </svg>
                                </button>
                            </div>
                        </div>

                        <div class="p-6">
                            <template x-if="selectedFiles.length > 0">
                                <div>
                                    <!-- Upload Progress List -->
                                    <div class="space-y-3 mb-6">
                                        <template x-for="(fileInfo, index) in selectedFiles" :key="index">
                                            <div 
                                                class="flex items-center gap-3 p-3 border rounded-lg"
                                                :class="{
                                                    'bg-green-50 border-green-200': fileInfo.status === 'success',
                                                    'bg-red-50 border-red-200': fileInfo.status === 'error',
                                                    'bg-blue-50 border-blue-200': fileInfo.status === 'uploading',
                                                    'bg-gray-50 border-gray-200': fileInfo.status === 'pending'
                                                }"
                                            >
                                                <!-- Status Icon -->
                                                <template x-if="fileInfo.status === 'uploading'">
                                                    <svg class="animate-spin h-5 w-5 text-blue-600 flex-shrink-0" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                                                        <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
                                                        <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                                                    </svg>
                                                </template>
                                                <template x-if="fileInfo.status === 'success'">
                                                    <svg class="w-5 h-5 text-green-600 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7" />
                                                    </svg>
                                                </template>
                                                <template x-if="fileInfo.status === 'error'">
                                                    <svg class="w-5 h-5 text-red-600 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12" />
                                                    </svg>
                                                </template>
                                                <template x-if="fileInfo.status === 'pending'">
                                                    <svg class="w-5 h-5 text-gray-400 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z" />
                                                    </svg>
                                                </template>
                                                
                                                <!-- File Info -->
                                                <div class="flex-1 min-w-0">
                                                    <p class="text-sm font-medium text-gray-900 truncate" x-text="fileInfo.name"></p>
                                                    <div class="flex items-center gap-2">
                                                        <p class="text-xs text-gray-500" x-text="(fileInfo.size / 1024).toFixed(1) + ' KB'"></p>
                                                        <template x-if="fileInfo.status === 'pending'">
                                                            <span class="text-xs text-gray-500">• {{ __('daylight::media.waiting') }}</span>
                                                        </template>
                                                        <template x-if="fileInfo.status === 'uploading'">
                                                            <span class="text-xs text-blue-700">• {{ __('daylight::media.uploading') }}</span>
                                                        </template>
                                                        <template x-if="fileInfo.status === 'success'">
                                                            <span class="text-xs text-green-700">• {{ __('daylight::media.upload_complete') }}</span>
                                                        </template>
                                                        <template x-if="fileInfo.status === 'error'">
                                                            <span class="text-xs text-red-700" x-text="'• ' + fileInfo.error"></span>
                                                        </template>
                                                    </div>
                                                </div>
                                            </div>
                                        </template>
                                    </div>
                                </div>
                            </template>

                            <template x-if="selectedFiles.length === 0">
                                <!-- Drag and Drop Area -->
                                <div
                                    class="border-2 border-dashed rounded-lg p-12 text-center transition"
                                    :class="isDragging ? 'border-sky-500 bg-sky-50' : 'border-gray-300'"
                                    x-on:dragover.prevent="isDragging = true"
                                    x-on:dragleave.prevent="isDragging = false"
                                    x-on:drop.prevent="handleDrop($event)"
                                >
                                    <svg class="mx-auto h-12 w-12 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 16a4 4 0 01-.88-7.903A5 5 0 1115.9 6L16 6a5 5 0 011 9.9M15 13l-3-3m0 0l-3 3m3-3v12" />
                                    </svg>
                                    <p class="mt-2 text-sm text-gray-600">{{ __('daylight::media.drag_drop_files') }}</p>
                                    <p class="mt-1 text-xs text-gray-500">{{ __('daylight::media.or') }}</p>
                                    
                                    <div class="mt-4">
                                        <input
                                            type="file"
                                            wire:model="uploads"
                                            x-ref="fileInput"
                                            multiple
                                            id="file-upload"
                                            class="hidden"
                                            x-on:change="handleFileSelect($event)"
                                        >
                                        <label for="file-upload" class="cursor-pointer">
                                            <x-daylight::buttons.primary type="button" x-on:click.prevent="$refs.fileInput.click()">
                                                {{ __('daylight::media.select_files') }}
                                            </x-daylight::buttons.primary>
                                        </label>
                                    </div>
                                </div>
                            </template>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </template>
</div>

