<?php

namespace Daylight\Core\Http\Controllers\Store;

use Daylight\Core\Http\Controllers\Controller;
use Daylight\Core\Http\Requests\Store\CustomerGroups\StoreCustomerGroupRequest;
use Daylight\Core\Http\Requests\Store\CustomerGroups\UpdateCustomerGroupRequest;
use Daylight\Core\Models\CustomerGroup;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;
use Illuminate\View\View;

class CustomerGroupController extends Controller implements HasMiddleware
{
    public static function middleware(): array
    {
        return [
            new Middleware('can:create:customer-groups', only: ['create', 'store']),
            new Middleware('can:read:customer-groups', only: ['index', 'show']),
            new Middleware('can:update:customer-groups', only: ['edit', 'update']),
            new Middleware('can:delete:customer-groups', only: ['destroy']),
        ];
    }

    public function index(): View
    {
        return view('daylight::store.customer-groups.index');
    }

    public function create(): View
    {
        return view('daylight::store.customer-groups.create');
    }

    public function store(StoreCustomerGroupRequest $request)
    {
        CustomerGroup::create($request->validated());

        return redirect()->route('daylight.store.customer-groups.index')
            ->with('success', __('Customer group has been created.'));
    }

    public function edit(CustomerGroup $customerGroup)
    {
        return view('daylight::store.customer-groups.edit', [
            'customerGroup' => $customerGroup,
        ]);
    }

    public function update(UpdateCustomerGroupRequest $request, CustomerGroup $customerGroup)
    {
        $customerGroup->update($request->validated());

        return redirect()->route('daylight.store.customer-groups.index')
            ->with('success', __('Customer group has been updated.'));
    }

    public function destroy(CustomerGroup $customerGroup)
    {
        $customerGroup->delete();

        return redirect()->route('daylight.store.customer-groups.index')
            ->with('success', __('Customer group has been deleted.'));
    }
}
