<?php

namespace Daylight\Core\Modules\Admin;

use Daylight\Core\Modules\Admin\Entities\Divider;
use Daylight\Core\Modules\Admin\Entities\Page;
use Daylight\Core\Modules\Admin\Entities\Script;
use Daylight\Core\Modules\Admin\Entities\Shortcut;
use Daylight\Core\Modules\Admin\Entities\Stylesheet;
use Daylight\Core\Modules\Admin\Providers\AssetServiceProvider;
use Daylight\Core\Modules\Admin\Providers\NavigationServiceProvider;
use Daylight\Core\Modules\Admin\Providers\ShortcutServiceProvider;
use Daylight\Core\Modules\Admin\Providers\WidgetServiceProvider;
use Daylight\Core\Modules\Admin\Registries\AssetRegistry;
use Daylight\Core\Modules\Admin\Registries\NavigationRegistry;
use Daylight\Core\Modules\Admin\Registries\ShortcutRegistry;
use Daylight\Core\Modules\Admin\Registries\WidgetRegistry;
use Daylight\Core\Modules\Admin\Widgets\CustomerCount;
use Illuminate\Support\ServiceProvider;

class AdminServiceProvider extends ServiceProvider
{
    public function register(): void
    {
        $this->app->register(AssetServiceProvider::class);
        $this->app->register(NavigationServiceProvider::class);
        $this->app->register(WidgetServiceProvider::class);
        $this->app->register(ShortcutServiceProvider::class);

        $this
            ->registerAssets()
            ->registerAdminNavigation()
            ->registerWidgets()
            ->registerShortcuts();
    }

    public function registerAssets(): self
    {
        AssetRegistry::addAsset(
            asset: Stylesheet::create()->setPath(daylightAsset('resources/assets/css/app.scss'))
        );

        AssetRegistry::addAsset(
            asset: Script::create()->setPath(daylightAsset('resources/assets/js/app.js'))->setDefer(true)
        );

        return $this;
    }

    public function registerAdminNavigation(): self
    {
        NavigationRegistry::addPage(
            Page::create()
                ->setOrder(100)
                ->setKey('dashboard')
                ->setLabel('Dashboard')
                ->setPermission('view:administration')
                ->setIcon('o-home')
                ->setRoute('daylight.index')
        );

        NavigationRegistry::addPage(
            Page::create()
                ->setOrder(200)
                ->setKey('store')
                ->setLabel('Store')
                ->setPermission('read:orders')
                ->setRoute('daylight.store.orders.index')
                ->setIcon('o-shopping-bag')
                ->setChildren([
                    Page::create()
                        ->setKey('orders')
                        ->setLabel('Orders')
                        ->setPermission('read:orders')
                        ->setRoute('daylight.store.orders.index')
                        ->setOrder(100),
                    Page::create()
                        ->setKey('customers')
                        ->setLabel('Customers')
                        ->setPermission('read:customers')
                        ->setRoute('daylight.store.customers.index')
                        ->setOrder(200),
                    Page::create()
                        ->setKey('customer_groups')
                        ->setLabel('Customer Groups')
                        ->setPermission('read:customer-groups')
                        ->setRoute('daylight.store.customer-groups.index')
                        ->setOrder(300),
                    Page::create()
                        ->setKey('users')
                        ->setLabel('Users')
                        ->setPermission('read:users')
                        ->setRoute('daylight.store.users.index')
                        ->setOrder(400),
                    Page::create()
                        ->setKey('products')
                        ->setLabel('Products')
                        ->setPermission('read:products')
                        ->setRoute('daylight.store.products.index')
                        ->setOrder(500),
                    Page::create()
                        ->setKey('variants')
                        ->setLabel('Variants')
                        ->setPermission('read:variants')
                        ->setRoute('daylight.store.variants.index')
                        ->setOrder(500),
                    Page::create()
                        ->setKey('brands')
                        ->setLabel('Brands')
                        ->setPermission('read:brands')
                        ->setRoute('daylight.store.brands.index')
                        ->setOrder(600),
                    Page::create()
                        ->setKey('categories')
                        ->setLabel('Categories')
                        ->setPermission('read:categories')
                        ->setRoute('daylight.store.categories.index')
                        ->setOrder(700),
                    Page::create()
                        ->setKey('labels')
                        ->setLabel('Labels')
                        ->setPermission('read:labels')
                        ->setRoute('daylight.store.labels.index')
                        ->setOrder(800),
                    Page::create()
                        ->setKey('attributes')
                        ->setLabel('Attributes')
                        ->setPermission('read:attributes')
                        ->setRoute('daylight.store.attributes.index')
                        ->setOrder(900),
                ])
        );

        NavigationRegistry::addPage(
            Page::create()
                ->setKey('content')
                ->setLabel('Content')
                ->setIcon('o-document-duplicate')
                ->setPermission('read:pages')
                ->setRoute('daylight.content.pages.index')
                ->setOrder(300)
                ->setChildren([
                    Page::create()
                        ->setKey('pages')
                        ->setLabel('Pages')
                        ->setPermission('read:pages')
                        ->setRoute('daylight.content.pages.index')
                        ->setOrder(100),
                ]),
        );

        NavigationRegistry::addPage(
            Page::create()
                ->setOrder(1000)
                ->setKey('settings')
                ->setLabel('Settings')
                ->setRoute('daylight.settings.general.index')
                ->setPermission('read:settings')
                ->setIcon('o-cog-6-tooth')
                ->setChildren([
                    Page::create()
                        ->setKey('general')
                        ->setLabel('General')
                        ->setPermission('read:settings')
                        ->setRoute('daylight.settings.general.index')
                        ->setOrder(100),
                    Divider::create()
                        ->setLabel('Store')
                        ->setOrder(150),
                    Page::create()
                        ->setKey('countries')
                        ->setLabel('Countries')
                        ->setPermission('read:settings')
                        ->setRoute('daylight.settings.countries.index')
                        ->setOrder(200),
                    Page::create()
                        ->setKey('shipping')
                        ->setLabel('Shipping')
                        ->setPermission('read:settings')
                        ->setRoute('daylight.settings.shipping-zones.index')
                        ->setOrder(300),
                    Divider::create()
                        ->setLabel('Advanced Settings')
                        ->setOrder(350),
                    Page::create()
                        ->setKey('tax_classes')
                        ->setLabel('Tax Classes')
                        ->setPermission('read:advanced-settings')
                        ->setRoute('daylight.settings.tax-classes.index')
                        ->setOrder(400),
                    Page::create()
                        ->setKey('shipping_classes')
                        ->setLabel('Shipping Classes')
                        ->setPermission('read:advanced-settings')
                        ->setRoute('daylight.settings.shipping-classes.index')
                        ->setOrder(500),
                    Page::create()
                        ->setKey('roles_permissions')
                        ->setLabel('Roles & Permissions')
                        ->setPermission('read:advanced-settings')
                        ->setRoute('daylight.settings.roles.index')
                        ->setOrder(600),
                ])
        );

        return $this;
    }

    public function registerWidgets(): self
    {
        WidgetRegistry::addWidget(
            new CustomerCount
        );

        return $this;
    }

    public function registerShortcuts(): self
    {
        ShortcutRegistry::addShortcut(
            Shortcut::create()
                ->setKey('orders')
                ->setColor('bg-pink-500')
                ->setIcon('o-shopping-cart')
                ->setLabel('View latest orders')
                ->setDescription('Quickly view the latest orders and manage them.')
                ->setRoute('daylight.store.orders.index')
                ->setPermission('read:orders')
        );

        return $this;
    }
}
