<?php

namespace App\Http\Controllers;

use Daylight\Core\Models\Enums\ProductStatus;
use Daylight\Core\Models\Translations\ProductTranslation;
use Daylight\Core\Modules\MediaLibrary\Models\Media;
use Daylight\Core\Modules\Multilingualism\Entities\LocalizedRoute;
use Illuminate\View\View;

class ProductController extends Controller
{
    public function index(): View
    {
        $localizedRoute = LocalizedRoute::create()
            ->setRoute('products.index')
            ->staticRoute();

        return view('products.index', [
            'localizedRoute' => $localizedRoute,
        ]);
    }

    public function show(string $slug): View
    {
        $product = ProductTranslation::query()
            ->with([
                'product.translations',
                'product.defaultTranslation',
                'product.media',
                'product.labels.translations' => function ($query) {
                    $query->where('locale', app()->getLocale());
                },
                'product.attributeValues.attribute.translations' => function ($query) {
                    $query->where('locale', app()->getLocale());
                },
                'product.variants.media',
                'product.attributeValues' => function ($query) {
                    $query
                        ->withWhereHas('attribute', function ($query) {
                            $query->whereVisible(1);
                        })
                        ->with([
                            'attribute.translations' => function ($query) {
                                $query->where('locale', app()->getLocale());
                            },
                            'translations' => function ($query) {
                                $query->where('locale', app()->getLocale());
                            },
                        ]);
                },
                'product.upSells' => function ($query) {
                    $query->with([
                        'target.translations' => function ($query) {
                            $query->where('locale', app()->getLocale());
                        },
                        'target.media',
                    ]);
                },
            ])
            ->whereHas('product', function ($query) {
                $query->where('status', ProductStatus::PUBLISHED);
            })
            ->where('slug', $slug)
            ->where('locale', app()->getLocale())
            ->firstOrFail();

        $localizedRoute = LocalizedRoute::create()
            ->setRoute('products.show', $product->slug)
            ->setDefaultTranslation($product->product->defaultTranslation)
            ->setTranslations($product->product->translations);

        return view('products.show', [
            'product' => $product,
            'localizedRoute' => $localizedRoute,
        ]);
    }

    public function downloadImages(string $slug)
    {
        $files = ProductTranslation::query()
            ->where('slug', $slug)
            ->where('locale', app()->getLocale())
            ->firstOrFail()
            ->product
            ->variants
            ->flatMap(fn ($variant) => $variant->media)
            ->map(fn (Media $media) => $media->handle)
            ->unique();

        return redirect()->to(sprintf(
            'https://upcdn.io/W142igx/archive%s?m=archive&%s',
            $files->first(),
            $files->skip(1)->map(fn ($file) => 'file='.$file)->join('&'),
        ));
    }

    public function download(string $slug)
    {
        $files = ProductTranslation::query()
            ->where('slug', $slug)
            ->where('locale', app()->getLocale())
            ->firstOrFail()
            ->product
            ->media
            ->map(fn (Media $media) => $media->handle)
            ->unique();

        return redirect()->to(sprintf(
            'https://upcdn.io/W142igx/archive%s?m=archive&%s',
            $files->first(),
            $files->skip(1)->map(fn ($file) => 'file='.$file)->join('&'),
        ));
    }
}
