<?php

namespace App\Daylight\Http\Controllers\Content;

use App\Daylight\Http\Requests\Content\Post\StorePostRequest;
use App\Daylight\Http\Requests\Content\Post\UpdatePostRequest;
use App\Http\Controllers\Controller;
use App\Models\Post;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;

class PostController extends Controller implements HasMiddleware
{
    public static function middleware(): array
    {
        return [
            new Middleware('can:create:pages', only: ['create', 'store']),
            new Middleware('can:read:pages', only: ['index', 'show']),
            new Middleware('can:update:pages', only: ['edit', 'update']),
            new Middleware('can:delete:pages', only: ['destroy']),
        ];
    }

    public function index()
    {
        return view('daylight.content.posts.index');

    }

    public function create()
    {
        return view('daylight.content.posts.create');
    }

    public function store(StorePostRequest $request)
    {
        $post = Post::create();

        $post->translations()->create([
            'locale' => $request->input('locale'),
            'name' => $request->input('name'),
            'slug' => $request->input('slug'),
            'meta_title' => $request->input('meta_title'),
            'meta_description' => $request->input('meta_description'),
            'robots' => $request->input('robots'),
            'content' => $request->input('content'),
        ]);

        $media = collect($request->get('image'))
            ->mapWithKeys(function ($media, $order) {
                return [$media['id'] => [
                    'order' => $order,
                    'properties' => [
                        'field' => 'image',
                    ],
                ]];
            });

        $post->media()->sync($media);

        return redirect()
            ->route('daylight.content.posts.edit', [
                'post' => $post,
                'locale' => $request->get('locale'),
            ])
            ->with('success', __('Post created'));
    }

    public function edit(Post $post)
    {
        $post->load([
            'translations' => function ($query) {
                $query->where('locale', request()->get('locale', app()->getLocale()));
            },
        ]);

        return view('daylight.content.posts.edit', [
            'post' => $post,
        ]);
    }

    public function update(UpdatePostRequest $request, Post $post)
    {
        $post->translations()->updateOrCreate(
            [
                'locale' => $request->input('locale'),
            ],
            [
                'name' => $request->input('name'),
                'slug' => $request->input('slug'),
                'meta_title' => $request->input('meta_title'),
                'meta_description' => $request->input('meta_description'),
                'robots' => $request->input('robots'),
                'content' => $request->input('content'),
            ],
        );

        $media = collect($request->get('image'))
            ->mapWithKeys(function ($media, $order) {
                return [$media['id'] => [
                    'order' => $order,
                    'properties' => [
                        'field' => 'image',
                    ],
                ]];
            });

        $post->media()->sync($media);

        return redirect()
            ->route('daylight.content.posts.edit', [
                'post' => $post,
                'locale' => $request->get('locale'),
            ])
            ->with('success', __('Post updated'));
    }

    public function destroy(Post $post)
    {
        $post->delete();

        return redirect()->route('daylight.content.posts.index')
            ->with('success', __('Post deleted'));
    }
}
