<?php

namespace App\Daylight\Http\Controllers\Content;

use App\Daylight\Http\Requests\Content\Questions\StoreQuestionRequest;
use App\Daylight\Http\Requests\Content\Questions\UpdateQuestionRequest;
use App\Http\Controllers\Controller;
use App\Models\Question;
use App\Models\QuestionCategory;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;

class QuestionController extends Controller implements HasMiddleware
{
    public static function middleware(): array
    {
        return [
            new Middleware('can:create:pages', only: ['create', 'store']),
            new Middleware('can:read:pages', only: ['index', 'show']),
            new Middleware('can:update:pages', only: ['edit', 'update']),
            new Middleware('can:delete:pages', only: ['destroy']),
        ];
    }

    public function index()
    {
        return view('daylight.content.questions.index');

    }

    public function create()
    {
        return view('daylight.content.questions.create', [
            'categories' => QuestionCategory::with('translations')->get(),
        ]);
    }

    public function store(StoreQuestionRequest $request)
    {
        $question = Question::create([
            'question_category_id' => $request->input('question_category_id'),
        ]);

        $question->translations()->create([
            'locale' => $request->input('locale'),
            'name' => $request->input('name'),
            'content' => $request->input('content'),
        ]);

        return redirect()
            ->route('daylight.content.questions.edit', [
                'question' => $question,
                'locale' => $request->get('locale'),
            ])
            ->with('success', __('Question created'));
    }

    public function edit(Question $question)
    {
        $question->load([
            'translations' => function ($query) {
                $query->where('locale', request()->get('locale', app()->getLocale()));
            },
        ]);

        return view('daylight.content.questions.edit', [
            'question' => $question,
            'categories' => QuestionCategory::with('translations')->get(),
        ]);
    }

    public function update(UpdateQuestionRequest $request, Question $question)
    {
        $question->update([
            'question_category_id' => $request->input('question_category_id'),
        ]);

        $question->translations()->updateOrCreate(
            [
                'locale' => $request->input('locale'),
            ],
            [
                'name' => $request->input('name'),
                'content' => $request->input('content'),
            ],
        );

        return redirect()
            ->route('daylight.content.questions.edit', [
                'question' => $question,
                'locale' => $request->get('locale'),
            ])
            ->with('success', __('Question updated'));
    }

    public function destroy(Question $question)
    {
        $question->delete();

        return redirect()->route('daylight.content.questions.index')
            ->with('success', __('Question deleted'));
    }
}
