<?php

namespace App\Http\Controllers\Account;

use App\Http\Controllers\Controller;
use App\Http\Requests\Account\Addresses\StoreAddressRequest;
use App\Http\Requests\Account\Addresses\UpdateAddressRequest;
use Daylight\Core\Models\Address;
use Daylight\Core\Models\Country;
use Daylight\Core\Modules\Multilingualism\Entities\LocalizedRoute;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;

class AddressController extends Controller
{
    public function index(): View
    {
        auth()->user()->load(['customer.billingAddresses', 'customer.shippingAddresses']);

        $localizedRoute = LocalizedRoute::create()
            ->setRoute('account.addresses.index')
            ->staticRoute();

        return view('account.addresses.index', [
            'billingAddresses' => auth()->user()->customer->billingAddresses,
            'shippingAddresses' => auth()->user()->customer->shippingAddresses,
            'localizedRoute' => $localizedRoute,
        ]);
    }

    public function create(): View|RedirectResponse
    {
        $localizedRoute = LocalizedRoute::create()
            ->setRoute('account.addresses.create')
            ->staticRoute();

        return view('account.addresses.create', [
            'localizedRoute' => $localizedRoute,
            'countries' => Country::query()
                ->select(['id', 'code'])
                ->get()
                ->mapWithKeys(fn ($country) => [$country->id => $country->name]),
        ]);
    }

    public function store(StoreAddressRequest $request)
    {
        $address = auth()->user()->customer->addresses()->create($request->validated());

        if ($request->default) {
            $address->customer->addresses()
                ->whereNotIn('id', [$address->id])
                ->whereType($address->type)
                ->update(['default' => false]);
        }

        return redirect()
            ->translatedRoute('account.addresses.index')
            ->with('success', __('Address created successfully.'));
    }

    public function edit(Address $address): View|RedirectResponse
    {
        if ($address->customer_id !== auth()->user()->customer_id) {
            return redirect()->translatedRoute('account.addresses.index');
        }

        $localizedRoute = LocalizedRoute::create()
            ->setRoute('account.addresses.edit', $address->id)
            ->staticRoute();

        return view('account.addresses.edit', [
            'localizedRoute' => $localizedRoute,
            'address' => $address,
            'countries' => Country::query()
                ->select(['id', 'code'])
                ->get()
                ->mapWithKeys(fn ($country) => [$country->id => $country->name]),
        ]);
    }

    public function update(Address $address, UpdateAddressRequest $request): RedirectResponse
    {
        $address->update($request->validated());

        if ($request->default) {
            $address->customer->addresses()
                ->whereNotIn('id', [$address->id])
                ->whereType($address->type)
                ->update(['default' => false]);
        }

        return redirect()
            ->translatedRoute('account.addresses.index')
            ->with('success', __('Address updated successfully.'));
    }

    public function destroy(Address $address): RedirectResponse
    {
        if ($address->customer_id !== auth()->user()->customer_id) {
            return redirect()->translatedRoute('account.addresses.index');
        }

        $address->delete();

        return redirect()
            ->translatedRoute('account.addresses.index')
            ->with('success', __('Address deleted successfully.'));
    }
}
