<?php

namespace App\Providers;

use App\Models\QuestionCategory;
use App\Models\Testimonial;
use Daylight\Core\Modules\Multilingualism\Multilingualism;
use Daylight\Core\Modules\PageBuilder\Models\Component;
use Illuminate\Auth\Middleware\Authenticate;
use Illuminate\Auth\Notifications\ResetPassword;
use Illuminate\Auth\Notifications\VerifyEmail;
use Illuminate\Http\Request;
use Illuminate\Pagination\Paginator;
use Illuminate\Routing\Redirector;
use Illuminate\Routing\UrlGenerator;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\ServiceProvider;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     */
    public function register(): void
    {
        Component::resolveRelationUsing('testimonials', function (Component $component) {
            return $component->morphedByMany(Testimonial::class, 'relatable');
        });

        Component::resolveRelationUsing('questionCategories', function (Component $component) {
            return $component->morphedByMany(QuestionCategory::class, 'relatable');
        });

        Redirector::macro('translatedRoute', function ($route, $parameters = [], $status = 302, $headers = []) {
            return redirect(route(app()->getLocale().'_'.$route, $parameters), $status, $headers);
        });

        UrlGenerator::macro('temporarySignedTranslatedRoute', function ($name, $expiration, $parameters = [], $absolute = true) {
            return $this->temporarySignedRoute(app()->getLocale().'_'.$name, $expiration, $parameters, $absolute);
        });

        Request::macro('translatedRouteIs', function ($route) {
            return $this->routeIs(app()->getLocale().'_'.$route);
        });

        VerifyEmail::createUrlUsing(function ($notifiable) {
            return URL::temporarySignedTranslatedRoute(
                'verification.verify',
                Carbon::now()->addMinutes(Config::get('auth.verification.expire', 60)),
                [
                    'id' => $notifiable->getKey(),
                    'hash' => sha1($notifiable->getEmailForVerification()),
                ]
            );
        });

        Authenticate::redirectUsing(function () {
            return translatedRoute('login');
        });

        ResetPassword::createUrlUsing(function ($notifiable, string $token) {
            return url(translatedRoute('password.reset', [
                'token' => $token,
                'email' => $notifiable->getEmailForPasswordReset(),
            ], app()->getLocale()));
        });
    }

    /**
     * Bootstrap any application services.
     */
    public function boot(): void
    {
        Paginator::defaultView('components.partials.pagination');

        if (Multilingualism::getAvailableLocales()->contains(request()->segment(1))) {
            App::setLocale(request()->segment(1));

            config([
                'multilingualism.locale_prefix' => request()->segment(1),
            ]);
        }
    }
}
